package com.dji.sample.manage.controller;

import com.dji.sample.common.model.CustomClaim;
import com.dji.sample.manage.model.dto.WorkspaceDTO;
import com.dji.sample.manage.model.param.searchParam.WorkspaceSearchParam;
import com.dji.sample.manage.service.IWorkspaceService;
import com.dji.sdk.common.HttpResultResponse;
import com.dji.sdk.common.PaginationData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Optional;

import static com.dji.sample.common.util.SecurityUtils.*;
import static com.dji.sample.component.AuthInterceptor.TOKEN_CLAIM;

/**
 * @author sean.zhou
 * @version 0.1
 * @date 2021/11/23
 */
@RestController
@RequestMapping("${url.manage.prefix}${url.manage.version}/workspaces")
public class WorkspaceController {

    @Autowired
    private IWorkspaceService workspaceService;

    /**
     * Gets information about the workspace that the current user is in.
     * @param request
     * @return
     */
    @GetMapping("/current")
    public HttpResultResponse getCurrentWorkspace(HttpServletRequest request) {
        CustomClaim customClaim = (CustomClaim)request.getAttribute(TOKEN_CLAIM);
        Optional<WorkspaceDTO> workspaceOpt = workspaceService.getWorkspaceByWorkspaceId(customClaim.getWorkspaceId());

        return workspaceOpt.isEmpty() ? HttpResultResponse.error() : HttpResultResponse.success(workspaceOpt.get());
    }

    @GetMapping("/getOne")
    public HttpResultResponse getWorkspace(WorkspaceSearchParam param) {

        // 判断权限 系统管理员才允许全查
        if (!aboveSysAdminRole()) {
            param.setRoleWorkspaceId(getWorkspaceId());
        }

        WorkspaceDTO workspace = workspaceService.getWorkspace(param);

        return HttpResultResponse.success(workspace);
    }

    @GetMapping("/page")
    public HttpResultResponse getWorkspacePage(WorkspaceSearchParam param,
                                               @RequestParam(defaultValue = "1") Long page,
                                               @RequestParam(value = "page_size", defaultValue = "50") Long pageSize) {

        // 判断权限 系统管理员才允许全查
        if (!aboveSysAdminRole()) {
            param.setRoleWorkspaceId(getWorkspaceId());
        }

        PaginationData<WorkspaceDTO> workspacePage = workspaceService.getWorkspacePage(param, page, pageSize);

        return HttpResultResponse.success(workspacePage);
    }

    @PostMapping("/delete")
    public HttpResultResponse delWorkspace(@RequestParam(name = "id") Integer id) {

        // 判断权限 系统管理员才允许修改
        aboveSysAdminRoleAndThrowError();

        boolean isDel = workspaceService.delWorkspaceById(id);

        return isDel ? HttpResultResponse.success() : HttpResultResponse.error();
    }

    @PostMapping("/edit")
    public HttpResultResponse editWorkspace(@RequestBody WorkspaceDTO workspaceDTO) {

        // 判断权限 管理员才允许修改
        aboveAdminRoleAndThrowError();

        boolean isEdit = workspaceService.editWorkspace(workspaceDTO);

        return isEdit ? HttpResultResponse.success() : HttpResultResponse.error();
    }

    @PostMapping("/add")
    public HttpResultResponse addWorkspace(@RequestBody WorkspaceDTO workspaceDTO) {

        // 判断权限 系统管理员才允许新增
        aboveSysAdminRoleAndThrowError();

        WorkspaceDTO workspace = workspaceService.addWorkspace(workspaceDTO);

        return HttpResultResponse.success(workspace);
    }

}