-- 1. 组织表
CREATE TABLE `manage_org` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `org_id` VARCHAR(64) NOT NULL UNIQUE COMMENT '组织 UUID',
  `org_name` VARCHAR(64) NOT NULL COMMENT '组织名称',
  `org_desc` VARCHAR(256) DEFAULT '' COMMENT '组织描述',
  `workspace_id` VARCHAR(64) NOT NULL COMMENT '统一 workspace ID，所有 org 共享同一个工作空间',
  `create_time` BIGINT NOT NULL,
  `update_time` BIGINT NOT NULL,
  `creator_id` VARCHAR(64) COMMENT '创建者ID',
  `creator_name` VARCHAR(64) COMMENT '创建者名称',
  `updater_id` VARCHAR(64) COMMENT '更新者ID',
  `updater_name` VARCHAR(64) COMMENT '更新者名称',
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='组织信息表';

-- 2. 用户组织关系表
CREATE TABLE `manage_user_org` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id` VARCHAR(64) NOT NULL COMMENT '用户 ID',
  `org_id` VARCHAR(64) NOT NULL COMMENT '组织 ID',
  `role_type` SMALLINT NOT NULL COMMENT '角色类型（100:SUPER_ADMIN, 50:ORG_ADMIN, 3:FLEET_ADMIN, 2:PILOT, 1:USER）',
  `status` TINYINT(1) DEFAULT 1,
  `create_time` BIGINT NOT NULL,
  `update_time` BIGINT NOT NULL,
  `creator_id` VARCHAR(64) COMMENT '创建者ID',
  `creator_name` VARCHAR(64) COMMENT '创建者名称',
  `updater_id` VARCHAR(64) COMMENT '更新者ID',
  `updater_name` VARCHAR(64) COMMENT '更新者名称',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_user_org` (`user_id`, `org_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='用户与组织关系（含角色）';

-- 3. 设备组织归属表（支持设备所属组织和共享）
CREATE TABLE `manage_device_org` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `device_id` INT NOT NULL COMMENT '设备 ID',
  `org_id` VARCHAR(64) NOT NULL COMMENT '组织 ID',
  `is_shared` TINYINT(1) DEFAULT 0 COMMENT '是否共享给其他组织（0: 否，1: 是）',
  `create_time` BIGINT NOT NULL DEFAULT 0 COMMENT '创建时间',
  `update_time` BIGINT NOT NULL DEFAULT 0 COMMENT '更新时间',
  `creator_id` VARCHAR(64) COMMENT '创建者ID',
  `creator_name` VARCHAR(64) COMMENT '创建者名称',
  `updater_id` VARCHAR(64) COMMENT '更新者ID',
  `updater_name` VARCHAR(64) COMMENT '更新者名称',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_device_org` (`device_id`, `org_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='设备与组织关系表';

-- 4. 修改 manage_user 表字段
-- 添加超级管理员标志、联系信息，并统一角色字段类型
-- 注意：在新架构中，我们将同时保留用户的 workspace_id 字段以实现平滑过渡
ALTER TABLE `manage_user`
  ADD COLUMN `org_id` VARCHAR(64) DEFAULT NULL COMMENT '用户主要关联的组织ID';

ALTER TABLE `manage_user`
  ADD COLUMN `is_super_admin` TINYINT(1) NOT NULL DEFAULT 0 COMMENT '是否为运营方超级管理员(0:否,1:是)';

ALTER TABLE `manage_user`
  ADD COLUMN `email` VARCHAR(64) DEFAULT '' COMMENT '用户登录邮箱';

ALTER TABLE `manage_user`
  ADD COLUMN `phone` VARCHAR(20) DEFAULT '' COMMENT '用户手机号';

ALTER TABLE `manage_user`
  ADD COLUMN `creator_id` VARCHAR(64) COMMENT '创建者ID';

ALTER TABLE `manage_user`
  ADD COLUMN `creator_name` VARCHAR(64) COMMENT '创建者名称';

ALTER TABLE `manage_user`
  ADD COLUMN `updater_id` VARCHAR(64) COMMENT '更新者ID';

ALTER TABLE `manage_user`
  ADD COLUMN `updater_name` VARCHAR(64) COMMENT '更新者名称';

-- 修改字段 role_type 的定义
ALTER TABLE `manage_user`
  MODIFY COLUMN `role_type` SMALLINT NOT NULL DEFAULT 1 COMMENT '账户默认角色（100:SUPER_ADMIN, 50:ORG_ADMIN, 3:FLEET_ADMIN, 2:PILOT, 1:USER）';

-- 5. 修改 manage_fleet_user 表
-- 添加用户在机队中的角色字段（整型枚举）
-- ALTER TABLE `manage_fleet_user`
--   ADD COLUMN IF NOT EXISTS `role` SMALLINT NOT NULL DEFAULT 2 COMMENT '用户在该机队内的角色（3:FLEET_ADMIN, 2:PILOT）',
--   ADD COLUMN IF NOT EXISTS `creator_id` VARCHAR(64) COMMENT '创建者ID',
--   ADD COLUMN IF NOT EXISTS `creator_name` VARCHAR(64) COMMENT '创建者名称',
--   ADD COLUMN IF NOT EXISTS `updater_id` VARCHAR(64) COMMENT '更新者ID',
--   ADD COLUMN IF NOT EXISTS `updater_name` VARCHAR(64) COMMENT '更新者名称';
ALTER TABLE `manage_fleet_user`
  ADD COLUMN `updater_name` VARCHAR(64) COMMENT '更新者名称';

ALTER TABLE `manage_fleet_user`
  ADD COLUMN `creator_id` VARCHAR(64) COMMENT '创建者ID';

ALTER TABLE `manage_fleet_user`
  ADD COLUMN `creator_name` VARCHAR(64) COMMENT '创建者名称';

ALTER TABLE `manage_fleet_user`
  ADD COLUMN `updater_id` VARCHAR(64) COMMENT '更新者ID';

ALTER TABLE `manage_fleet_user`
  ADD COLUMN `updater_name` VARCHAR(64) COMMENT '更新者名称';

-- 6. 关于旧架构字段的处理

-- 说明：为了最小化修改，我们保留现有的 workspace_id 字段和相关表结构
-- 在新架构中，用户可以通过组织间接关联到工作空间，也可以保留直接关联
-- 用户同时属于组织和工作空间，组织属于工作空间

-- 注释：如果未来确定完全迁移到新架构，可以考虑删除这些结构
-- ALTER TABLE `manage_user` DROP COLUMN IF EXISTS `workspace_id`;
-- DROP TABLE IF EXISTS `manage_user_workspace`;

-- 添加索引以提高关联查询效率
ALTER TABLE `manage_user` 
  ADD INDEX IF NOT EXISTS `idx_workspace_id` (`workspace_id`),
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 7. 初始化数据（如果需要）

-- 7.1 创建默认工作空间
-- INSERT INTO `manage_workspace` (`workspace_id`, `workspace_name`, `create_time`, `update_time`) 
-- VALUES ('default-workspace-id', '默认工作空间', UNIX_TIMESTAMP() * 1000, UNIX_TIMESTAMP() * 1000);

-- 7.2 创建默认组织
-- INSERT INTO `manage_org` (`org_id`, `org_name`, `org_desc`, `workspace_id`, `create_time`, `update_time`, `creator_id`, `creator_name`, `updater_id`, `updater_name`)
-- VALUES (UUID(), '默认组织', '系统默认创建的组织', 'default-workspace-id', UNIX_TIMESTAMP() * 1000, UNIX_TIMESTAMP() * 1000, 'system', 'system', 'system', 'system');

-- 7.3 创建超级管理员用户
-- INSERT INTO `manage_user` (`user_id`, `username`, `password`, `user_type`, `role_type`, `workspace_id`, `org_id`, `is_super_admin`, `email`, `create_time`, `update_time`, `creator_id`, `creator_name`, `updater_id`, `updater_name`) 
-- VALUES (UUID(), 'superadmin', 'YOUR_ENCRYPTED_PASSWORD', 1, 100, 'default-workspace-id', (SELECT `org_id` FROM `manage_org` WHERE `org_name` = '默认组织' LIMIT 1), 1, 'admin@example.com', UNIX_TIMESTAMP() * 1000, UNIX_TIMESTAMP() * 1000, 'system', 'system', 'system', 'system');

-- 7.4 将超级管理员添加到默认组织
-- INSERT INTO `manage_user_org` (`user_id`, `org_id`, `role_type`, `status`, `create_time`, `update_time`, `creator_id`, `creator_name`, `updater_id`, `updater_name`)
-- SELECT u.user_id, o.org_id, 100, 1, UNIX_TIMESTAMP() * 1000, UNIX_TIMESTAMP() * 1000, 'system', 'system', 'system', 'system'
-- FROM `manage_user` u, `manage_org` o 
-- WHERE u.username = 'superadmin' AND o.org_name = '默认组织';

-- 7.5 保持 manage_user_workspace 表的内容同步（可选）
-- INSERT INTO `manage_user_workspace` (`user_id`, `workspace_id`, `role_type`, `status`, `create_time`, `update_time`)
-- SELECT u.user_id, u.workspace_id, u.role_type, 1, UNIX_TIMESTAMP() * 1000, UNIX_TIMESTAMP() * 1000
-- FROM `manage_user` u
-- WHERE u.username = 'superadmin';

-- 8. 为其他相关表添加 org_id 字段

-- 8.1 设备相关表
ALTER TABLE `manage_device` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '设备所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 8.2 媒体文件表
ALTER TABLE `media_file` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '媒体文件所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 8.3 航线相关表
ALTER TABLE `wayline_file` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '航线文件所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `wayline_job` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '航线任务所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 8.4 地图相关表
ALTER TABLE `map_group` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '地图组所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `map_group_element` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '地图元素所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `flight_area_file` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '飞行区域文件所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `device_flight_area` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '设备飞行区域所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 8.5 日志相关表
ALTER TABLE `logs_file` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '日志文件所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `logs_file_index` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '日志文件索引所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `manage_device_logs` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '设备日志所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `manage_device_hms` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '设备HMS所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `manage_device_payload` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '设备载荷所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

ALTER TABLE `manage_device_firmware` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '设备固件所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 8.6 机队相关表
-- 8.6.1 创建机队表（如果不存在）
CREATE TABLE IF NOT EXISTS `manage_fleet` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `fleet_id` VARCHAR(64) NOT NULL UNIQUE COMMENT '机队 UUID',
  `fleet_name` VARCHAR(64) NOT NULL COMMENT '机队名称',
  `fleet_desc` VARCHAR(256) DEFAULT '' COMMENT '机队描述',
  `org_id` VARCHAR(64) NOT NULL COMMENT '机队所属组织ID',
  `workspace_id` VARCHAR(64) NOT NULL COMMENT '机队所属工作空间ID',
  `create_time` BIGINT NOT NULL,
  `update_time` BIGINT NOT NULL,
  `creator_id` VARCHAR(64) COMMENT '创建者ID',
  `creator_name` VARCHAR(64) COMMENT '创建者名称',
  `updater_id` VARCHAR(64) COMMENT '更新者ID',
  `updater_name` VARCHAR(64) COMMENT '更新者名称',
  PRIMARY KEY (`id`),
  INDEX `idx_org_id` (`org_id`),
  INDEX `idx_workspace_id` (`workspace_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='机队信息表';

-- 8.6.2 创建机队设备关系表（如果不存在）
CREATE TABLE IF NOT EXISTS `manage_fleet_device` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `fleet_id` VARCHAR(64) NOT NULL COMMENT '机队 ID',
  `device_id` INT NOT NULL COMMENT '设备 ID',
  `device_sn` VARCHAR(64) NOT NULL COMMENT '设备 SN',
  `org_id` VARCHAR(64) NOT NULL COMMENT '关系所属组织ID',
  `create_time` BIGINT NOT NULL,
  `update_time` BIGINT NOT NULL,
  `creator_id` VARCHAR(64) COMMENT '创建者ID',
  `creator_name` VARCHAR(64) COMMENT '创建者名称',
  `updater_id` VARCHAR(64) COMMENT '更新者ID',
  `updater_name` VARCHAR(64) COMMENT '更新者名称',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_fleet_device` (`fleet_id`, `device_id`),
  INDEX `idx_org_id` (`org_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='机队与设备关系表';

-- 8.6.3 为已存在的 manage_fleet_user 表添加 org_id 字段
ALTER TABLE `manage_fleet_user` 
  ADD COLUMN IF NOT EXISTS `org_id` VARCHAR(64) DEFAULT NULL COMMENT '机队用户关系所属组织ID',
  ADD INDEX IF NOT EXISTS `idx_org_id` (`org_id`);

-- 9. 数据迁移脚本（可选）
-- 将现有数据根据 workspace_id 关联到默认组织

-- 9.1 为设备数据关联组织
-- UPDATE `manage_device` d 
-- SET d.org_id = (
--     SELECT o.org_id 
--     FROM `manage_org` o 
--     WHERE o.workspace_id = d.workspace_id 
--     AND o.org_name = '默认组织'
--     LIMIT 1
-- )
-- WHERE d.org_id IS NULL;

-- 9.2 为媒体文件关联组织
-- UPDATE `media_file` m 
-- SET m.org_id = (
--     SELECT o.org_id 
--     FROM `manage_org` o 
--     WHERE o.workspace_id = m.workspace_id 
--     AND o.org_name = '默认组织'
--     LIMIT 1
-- )
-- WHERE m.org_id IS NULL;

-- 9.3 为航线文件关联组织
-- UPDATE `wayline_file` w 
-- SET w.org_id = (
--     SELECT o.org_id 
--     FROM `manage_org` o 
--     WHERE o.workspace_id = w.workspace_id 
--     AND o.org_name = '默认组织'
--     LIMIT 1
-- )
-- WHERE w.org_id IS NULL;

-- 9.4 为航线任务关联组织
-- UPDATE `wayline_job` j 
-- SET j.org_id = (
--     SELECT o.org_id 
--     FROM `manage_org` o 
--     WHERE o.workspace_id = j.workspace_id 
--     AND o.org_name = '默认组织'
--     LIMIT 1
-- )
-- WHERE j.org_id IS NULL;

-- 9.5 为地图组关联组织
-- UPDATE `map_group` g 
-- SET g.org_id = (
--     SELECT o.org_id 
--     FROM `manage_org` o 
--     WHERE o.workspace_id = g.workspace_id 
--     AND o.org_name = '默认组织'
--     LIMIT 1
-- )
-- WHERE g.org_id IS NULL;

-- 注意：执行数据迁移前请备份数据库！