package com.dji.sample.manage.controller;

import com.dji.sample.common.model.CustomClaim;
import com.dji.sample.manage.model.dto.OrgDTO;
import com.dji.sample.manage.model.dto.OrgJoinDTO;
import com.dji.sample.manage.model.dto.WorkspaceDTO;
import com.dji.sample.manage.model.param.searchParam.OrgSearchParam;
import com.dji.sample.manage.model.param.searchParam.WorkspaceSearchParam;
import com.dji.sample.manage.service.IOrgService;
import com.dji.sample.manage.service.IWorkspaceService;
import com.dji.sdk.common.HttpResultResponse;
import com.dji.sdk.common.PaginationData;
import io.swagger.v3.oas.annotations.Operation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Objects;
import java.util.Optional;

import static com.dji.sample.common.util.SecurityUtils.*;
import static com.dji.sample.component.AuthInterceptor.TOKEN_CLAIM;

@RestController
@RequestMapping("${url.manage.prefix}${url.manage.version}/orgs")
public class OrgController {

    @Autowired
    private IOrgService orgService;

    /**
     * Gets information about the workspace that the current user is in.
     * @param request
     * @return
     */
    @GetMapping("/current")
    public HttpResultResponse getCurrentOrg(HttpServletRequest request) {
        CustomClaim customClaim = (CustomClaim)request.getAttribute(TOKEN_CLAIM);
        Optional<OrgDTO> orgOpt = orgService.getOrgByOrgId(customClaim.getOrgId());

        return orgOpt.isEmpty() ? HttpResultResponse.error() : HttpResultResponse.success(orgOpt.get());
    }

    @GetMapping("/getOne")
    public HttpResultResponse getWorkspace(OrgSearchParam param) {
        param.setWorkspaceId(getWorkspaceId());
        // 判断权限 系统管理员才允许全查
        if (!aboveSysAdminRole()) {
            param.setRoleUserId(getUserId());
        }

        OrgDTO org = orgService.getOrg(param);

        return HttpResultResponse.success(org);
    }

    @GetMapping("/page")
    public HttpResultResponse getOrgPage(OrgSearchParam param,
                                         @RequestParam(defaultValue = "1") Long page,
                                         @RequestParam(value = "page_size", defaultValue = "50") Long pageSize) {

        PaginationData<OrgDTO> workspacePage = orgService.getOrgPage(param, page, pageSize);

        return HttpResultResponse.success(workspacePage);
    }

    @PostMapping("/add")
    public HttpResultResponse addOrg(@RequestBody OrgDTO orgDTO) {

        // 判断权限 系统管理员才允许新增
        aboveSysAdminRoleAndThrowError();

        orgDTO.setWorkspaceId(getWorkspaceId());
        OrgDTO org = orgService.addOrg(orgDTO);

        return HttpResultResponse.success(org);
    }

    @PostMapping("/edit")
    public HttpResultResponse editOrg(@RequestBody OrgDTO orgDTO) {

        // 判断权限 管理员才允许修改
        aboveAdminRoleAndThrowError();

        orgDTO.setWorkspaceId(getWorkspaceId());
        boolean isEdit = orgService.editOrg(orgDTO);

        return isEdit ? HttpResultResponse.success() : HttpResultResponse.error();
    }

    @PostMapping("/delete")
    public HttpResultResponse delWorkspace(@RequestBody OrgDTO orgDTO) {

        // 判断权限 系统管理员才允许修改
        aboveSysAdminRoleAndThrowError();

        boolean isDel = orgService.delOrg(orgDTO);

        return isDel ? HttpResultResponse.success() : HttpResultResponse.error();
    }

    @PostMapping("/apply")
    public HttpResultResponse applyOrg(@RequestBody OrgJoinDTO orgDTO) {

        boolean isDel = orgService.applyOrg(orgDTO);

        return isDel ? HttpResultResponse.success() : HttpResultResponse.error();
    }

    @PostMapping("/joinByAdmin")
    public HttpResultResponse joinOrgByAdmin(@RequestBody OrgJoinDTO orgDTO) {

        boolean isDel = orgService.joinOrgByAdmin(orgDTO);

        return isDel ? HttpResultResponse.success() : HttpResultResponse.error();
    }

    /**
     * upload org logo
     * @param req
     * @param rsp
     * @return
     */
    @Operation(summary = "upload org logo", description = "upload org logo")
    @PostMapping("/uploadLogo")
    HttpResultResponse upload(MultipartFile file,
                              HttpServletRequest req, HttpServletResponse rsp) {
        if (Objects.isNull(file)) {
            return HttpResultResponse.error("No file received.");
        }
        aboveOrgAdminRoleAndThrowError();

        String logoUrl = orgService.saveLogo(file);
        return HttpResultResponse.success(logoUrl);
    }
    /**
     * Gets information about the workspace that the current user is in.
     * @param request
     * @return
     */
    @GetMapping("/getLogo")
    public HttpResultResponse getLogo(HttpServletRequest request) {

        String logoUrl = orgService.getLogo();
        return HttpResultResponse.success(logoUrl);
    }



}